<?php
/**
 * Plugin Name: Taxonomy Parent
 * Plugin URI: https://phoenixseo.de/blog/wordpress-plugin-taxonomy-parent
 * Description: Erstellt einen Gutenberg-Block, der den übergeordneten Begriff einer hierachischen Taxonomie als Link oder Text anzeigt.
 * Version: 1.4
 * Author: PHOENIXSEO.de Frank Pfabigan e.K.
 * Author URI: https://phoenixseo.de
 * License: GPL2
 * Text Domain: taxonomy-parent
 */

if (!defined('ABSPATH')) {
    exit;
}

// Hierarchische Taxonomien abrufen
function taxonomy_parent_get_hierarchical_taxonomies() {
    $taxonomies = get_taxonomies(array(
        'hierarchical' => true,
        'show_in_rest' => true
        ), 'objects');
    $options = [];

    foreach ($taxonomies as $taxonomy) {
        $options[] = [
            'label' => $taxonomy->labels->singular_name,
            'value' => $taxonomy->name
        ];
    }

    return $options;
}

// Block-Assets registrieren
function taxonomy_parent_register_block() {
    wp_register_script(
        'taxonomy-parent-editor-script',
        plugins_url('build/index.js', __FILE__),
        array('wp-blocks', 'wp-editor', 'wp-components', 'wp-i18n', 'wp-element'),
        filemtime(plugin_dir_path(__FILE__) . 'build/index.js')
    );

    wp_localize_script('taxonomy-parent-editor-script', 'taxonomyParentData', [
        'taxonomies' => taxonomy_parent_get_hierarchical_taxonomies()
    ]);

    register_block_type('taxonomy-parent/block', array(
        'editor_script'   => 'taxonomy-parent-editor-script',
        'render_callback' => 'taxonomy_parent_render_callback',
        'attributes'      => array(
            'taxonomy'  => array(
                'type'    => 'string',
                'default' => 'category'
            ),
            'prefixText' => array(
                'type'    => 'string',
                'default' => __('Übergeordneter Begriff:', 'taxonomy-parent')
            ),
            'asLink' => array(
                'type'    => 'boolean',
                'default' => true
            )
        )
    ));
}
add_action('init', 'taxonomy_parent_register_block');

// Server-Rendering-Funktion für den Block
/*
function taxonomy_parent_render_callback($attributes) {
    if (!is_singular()) {
        return '';
    }

    $post_id   = get_the_ID();
    $taxonomy  = $attributes['taxonomy'] ?? 'category';
    $prefix    = esc_html($attributes['prefixText'] ?? __('Übergeordneter Begriff:', 'taxonomy-parent'));
    $as_link   = $attributes['asLink'] ?? true;

    $terms = get_the_terms($post_id, $taxonomy);
    if (!$terms || is_wp_error($terms)) {
        return '';
    }

    foreach ($terms as $term) {
        if ($term->parent) {
            $parent_term = get_term($term->parent, $taxonomy);
            if ($parent_term && !is_wp_error($parent_term)) {
                $term_name = esc_html($parent_term->name);
                $term_link = get_term_link($parent_term);

                if ($as_link && !is_wp_error($term_link)) {
                    return '<div class="taxonomy-parent-block">' . $prefix . ' <a href="' . esc_url($term_link) . '">' . $term_name . '</a></div>';
                } else {
                    return '<div class="taxonomy-parent-block">' . $prefix . ' <strong>' . $term_name . '</strong></div>';
                }
            }
        }
    }

    return '<div class="taxonomy-parent-block">' . $prefix . ' ' . __('Kein übergeordneter Begriff gefunden.', 'taxonomy-parent') . '</div>';
}
*/
function taxonomy_parent_render_callback($attributes) {
    $prefix  = esc_html($attributes['prefixText'] ?? __('Übergeordneter Begriff:', 'taxonomy-parent'));
    $as_link = $attributes['asLink'] ?? true;
    $output  = '';

    // 🟢 1. FALL: SINGULÄRE BEITRAGSSEITE
    if (is_singular()) {
        $post_id  = get_the_ID();
        $taxonomy = $attributes['taxonomy'] ?? 'category';
        $terms    = get_the_terms($post_id, $taxonomy);

        if ($terms && !is_wp_error($terms)) {
            foreach ($terms as $term) {
                if ($term->parent) {
                    $parent_term = get_term($term->parent, $taxonomy);
                    if ($parent_term && !is_wp_error($parent_term)) {
                        $term_name = esc_html($parent_term->name);
                        $term_link = get_term_link($parent_term);

                        $output .= $prefix . ' ';
                        $output .= ($as_link && !is_wp_error($term_link)) ? '<a href="' . esc_url($term_link) . '">' . $term_name . '</a>' : '<strong>' . $term_name . '</strong>';
                    }
                }
            }
        }
    }
    
    // 🟢 2. FALL: TAXONOMIE-ARCHIVSEITE
    elseif (is_tax() || is_category() || is_tag()) {
        $taxonomy = get_queried_object();
        
        if ($taxonomy && isset($taxonomy->parent) && $taxonomy->parent) {
            $parent_term = get_term($taxonomy->parent, $taxonomy->taxonomy);
            if ($parent_term && !is_wp_error($parent_term)) {
                $term_name = esc_html($parent_term->name);
                $term_link = get_term_link($parent_term);

                $output .= $prefix . ' ';
                $output .= ($as_link && !is_wp_error($term_link)) ? '<a href="' . esc_url($term_link) . '">' . $term_name . '</a>' : '<strong>' . $term_name . '</strong>';
            }
        }
    }

    return $output ? '<div class="taxonomy-parent-block">' . $output . '</div>' : '';
}
