import { registerBlockType } from '@wordpress/blocks';
import { useBlockProps, InnerBlocks, InspectorControls } from '@wordpress/block-editor';
import { useEffect, useState } from '@wordpress/element';
import { PanelBody, RangeControl, TextControl, SelectControl } from '@wordpress/components';

registerBlockType('urbblock/unsplash-background', {
    title: 'URB-Block Unsplash Hintergrund',
    category: 'design',
    attributes: {
        imageUrl: { type: 'string', default: '' },
        alpha: { type: 'number', default: 1 },
        padding: { type: 'string', default: '20px' },
        backgroundPositionX: { type: 'string', default: 'center' },
        backgroundPositionY: { type: 'string', default: 'center' },
        widthPercentage: { type: 'number', default: 100 },
        height: { type: 'string', default: '300px' }
    },
    edit: ({ attributes, setAttributes }) => {
        const blockProps = useBlockProps();
        const [loading, setLoading] = useState(true);

        useEffect(() => {
            const apiKey = window.urbblockSettings?.apiKey;
            const collectionId = window.urbblockSettings?.collectionId || '4759555';
            if (!attributes.imageUrl && apiKey) {
                fetch(`https://api.unsplash.com/collections/${collectionId}/photos?per_page=30`, {
                    headers: { 'Authorization': `Client-ID ${apiKey}` }
                })
                    .then(response => response.json())
                    .then(data => {
                        const randomImage = data[Math.floor(Math.random() * data.length)]?.urls?.regular;
                        setAttributes({ imageUrl: randomImage });
                    })
                    .finally(() => setLoading(false));
            } else {
                setLoading(false);
            }
        }, []);

        return (
            <div {...blockProps} style={{
                backgroundImage: `url(${attributes.imageUrl})`,
                backgroundSize: 'cover',
                backgroundPosition: `${attributes.backgroundPositionX} ${attributes.backgroundPositionY}`,
                padding: attributes.padding,
                minHeight: attributes.height,
                width: `${attributes.widthPercentage}%`,
                opacity: attributes.alpha
            }}>
                <InspectorControls>
                    <PanelBody title="Hintergrund Transparenz">
                        <RangeControl
                            label="Alpha-Wert"
                            value={attributes.alpha}
                            onChange={(value) => setAttributes({ alpha: value })}
                            min={0.1}
                            max={1}
                            step={0.1}
                        />
                    </PanelBody>
                    <PanelBody title="Padding Einstellungen">
                        <TextControl
                            label="Padding (px oder em)"
                            value={attributes.padding}
                            onChange={(value) => setAttributes({ padding: value })}
                            placeholder="z.B. 20px oder 1em"
                        />
                    </PanelBody>
                    <PanelBody title="Hintergrundposition">
                        <SelectControl
                            label="Horizontale Position"
                            value={attributes.backgroundPositionX}
                            options={[
                                { label: 'Links', value: 'left' },
                                { label: 'Zentrum', value: 'center' },
                                { label: 'Rechts', value: 'right' }
                            ]}
                            onChange={(value) => setAttributes({ backgroundPositionX: value })}
                        />
                        <SelectControl
                            label="Vertikale Position"
                            value={attributes.backgroundPositionY}
                            options={[
                                { label: 'Oben', value: 'top' },
                                { label: 'Zentrum', value: 'center' },
                                { label: 'Unten', value: 'bottom' }
                            ]}
                            onChange={(value) => setAttributes({ backgroundPositionY: value })}
                        />
                    </PanelBody>
                    <PanelBody title="Blockgröße">
                        <RangeControl
                            label="Breite (%)"
                            value={attributes.widthPercentage}
                            onChange={(value) => setAttributes({ widthPercentage: value })}
                            min={0}
                            max={100}
                        />
                        <TextControl
                            label="Höhe (px oder em)"
                            value={attributes.height}
                            onChange={(value) => setAttributes({ height: value })}
                            placeholder="z.B. 300px oder 20em"
                        />
                    </PanelBody>
                </InspectorControls>
                {loading ? <p>Lädt...</p> : <InnerBlocks />}
            </div>
        );
    },
    save: ({ attributes }) => {
        const blockProps = useBlockProps.save();
        return (
            <div {...blockProps} style={{
                backgroundImage: `url(${attributes.imageUrl})`,
                backgroundSize: 'cover',
                backgroundPosition: `${attributes.backgroundPositionX} ${attributes.backgroundPositionY}`,
                padding: attributes.padding,
                minHeight: attributes.height,
                width: `${attributes.widthPercentage}%`,
                opacity: attributes.alpha
            }}>
                <InnerBlocks.Content />
            </div>
        );
    }
});
